<?php
// FILE: qr_process_dynamic.php - Memproses Absensi Siswa Menggunakan QR Dinamis Kantor

session_start();
require_once 'db.php'; 
date_default_timezone_set('Asia/Jakarta');

// === Konfigurasi (Wajib sama dengan office_qr_display.php) ===
$secret_key = "BAITULILMI_QR_SECRET_KEY_2025"; 
// refresh_interval harus sama dengan $refresh_interval di office_qr_display.php
$refresh_interval = 30; 
// =============================================================

// 1. Cek Login Siswa
if (!isset($_SESSION['is_siswa_logged_in']) || $_SESSION['is_siswa_logged_in'] !== true || !isset($_SESSION['siswa_id'])) {
    // Jika tidak login, alihkan ke halaman login siswa.
    header("Location: login_siswa.php?error=not_logged_in");
    exit;
}

$siswa_id = $_SESSION['siswa_id'];
// Token yang dipindai siswa akan dikirim melalui parameter GET dari scanner
$token_scanned = $_GET['token'] ?? null; 

$result = [
    'is_error' => true,
    'message' => 'Token tidak ditemukan atau sesi bermasalah.',
    'status_hadir' => 'Gagal'
];

if (!$token_scanned) {
    $result['message'] = 'Gagal memproses. Token absensi tidak terdeteksi dari hasil pindai.';
} else {
    try {
        // 2. Validasi Token Dinamis
        // Cek ke database apakah token valid dan belum kadaluarsa
        $stmt_token = $pdo->prepare("SELECT valid_until FROM absensi_token_kantor WHERE token_value = ? AND valid_until > NOW()");
        $stmt_token->execute([$token_scanned]);
        $token_valid = $stmt_token->fetch(PDO::FETCH_ASSOC);

        if (!$token_valid) {
            $result['message'] = 'Gagal Absen. Kode QR sudah kadaluarsa, sudah terlalu lama, atau tidak valid.';
        } else {
            // 3. Ambil Data Siswa (Untuk pesan hasil)
            $stmt_siswa = $pdo->prepare("SELECT nama, nisn FROM siswa WHERE siswa_id = ?");
            $stmt_siswa->execute([$siswa_id]);
            $siswa_profile = $stmt_siswa->fetch(PDO::FETCH_ASSOC);

            if (!$siswa_profile) {
                $result['message'] = "Error: Data Siswa tidak ditemukan.";
            } else {
                $tanggal_hari_ini = date('Y-m-d');
                $bulan_hari_ini = date('Y-m');
                
                // 4. Cek apakah siswa sudah absen hari ini
                $stmt_cek = $pdo->prepare("SELECT status FROM kehadiran WHERE siswa_id = ? AND tanggal = ?");
                $stmt_cek->execute([$siswa_id, $tanggal_hari_ini]);
                $status_kehadiran = $stmt_cek->fetchColumn();

                if ($status_kehadiran) {
                    $result['is_error'] = true;
                    $result['status_hadir'] = $status_kehadiran;
                    $result['message'] = "Perhatian! " . htmlspecialchars($siswa_profile['nama']) . " sudah absen hari ini ({$status_kehadiran}).";
                } else {
                    // 5. Masukkan data kehadiran
                    $stmt_insert = $pdo->prepare("INSERT INTO kehadiran (siswa_id, tanggal, status, bulan) VALUES (?, ?, 'H', ?)");
                    if ($stmt_insert->execute([$siswa_id, $tanggal_hari_ini, $bulan_hari_ini])) {
                        $result['is_error'] = false;
                        $result['status_hadir'] = 'H';
                        $result['message'] = "Berhasil! Kehadiran " . htmlspecialchars($siswa_profile['nama']) . " tercatat Hadir.";
                    } else {
                        $result['message'] = "Gagal! Terjadi kesalahan saat menyimpan data kehadiran.";
                    }
                }
                $result['profile'] = $siswa_profile; // Tambahkan profil untuk tampilan hasil
            }
        }
    } catch (PDOException $e) {
        $result['message'] = "Terjadi kesalahan pada database: " . $e->getMessage();
    }
}

$title = $result['is_error'] ? "Gagal Absen" : "Absensi Berhasil!";
$color = $result['is_error'] ? "red" : "green";
$result['waktu_scan'] = date('H:i:s');
?>

<!DOCTYPE html>
<html>
<head>
    <title><?= $title ?></title>
    <meta name='viewport' content='width=device-width, initial-scale=1.0'>
    <script src='https://cdn.tailwindcss.com'></script>
    <link rel='stylesheet' href='https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css'>
</head>
<body class='bg-gradient-to-br from-indigo-700 to-blue-800 flex items-center justify-center min-h-screen p-4'>
    <div class='bg-white p-8 rounded-2xl shadow-2xl text-center w-full max-w-sm'>
        <i class='fas fa-<?= $result['is_error'] ? 'times' : 'check' ?>-circle text-6xl text-<?= $color ?>-500 mb-4'></i>
        <h1 class='text-2xl font-extrabold text-indigo-700 mb-2'>STATUS ABSENSI</h1>
        <p class='text-lg font-semibold text-gray-700 mb-4'><?= htmlspecialchars($result['message']) ?></p>
        <?php if (isset($result['profile'])): ?>
            <h3 class='text-3xl font-extrabold text-indigo-700 mb-2'><?= htmlspecialchars($result['profile']['nama']) ?></h3>
            <p class='text-md text-gray-600 mb-4'>NISN: <?= htmlspecialchars($result['profile']['nisn'] ?? 'N/A') ?> (#<?= $siswa_id ?>)</p>
        <?php endif; ?>
        <p class='text-sm text-gray-500'>Waktu Absen: <?= htmlspecialchars($result['waktu_scan']) ?></p>
        <a href="akun_siswa.php" class='mt-6 inline-block px-6 py-2 bg-indigo-600 text-white rounded-lg hover:bg-indigo-700 transition duration-300'>Kembali ke Dashboard</a>
    </div>
</body>
</html>